<?php

/**
 * Class CompetitorModel. Clase CompetitorModel
 */
class CompetitorModel extends CI_Model


{
    const TABLE_NAME = 'Persona';
    const ALL_WEIGHTS = 0;

    /**
     * CompetitorModel constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Participaciones de competidores entre dos fechas indicando peso y preparado para
     * la paginacion
     * @param $fecha_i fecha de inicio de busqueda
     * @param $fecha_f fecha de fin de busqueda
     * @param $weight Identificador del peso
     * @param $perpage Numero de competidores por pagina
     * @param $offset Desplazamiento para la base de datos
     * @param $agegroup Identificador de grupo de edad
     * @return mixed
     */
    public function getCompetitorsByDate($fecha_i, $fecha_f, $weight, $perpage, $offset,$agegroup)
    {
        if($weight == self::ALL_WEIGHTS) {
            $sql = 'select c.*, p.*
                from persona c, participacion p ,peso pe 
                where c.dni=p.dniCompetidor and (p.fecha_participacion BETWEEN ? and ?) and p.idPeso = pe.idPeso and pe.idGrupoEdad = ?
                limit ? OFFSET ?';
            $offset = $offset == null ? 0 : $offset;
            $query = $this->db->query($sql, array($fecha_i, $fecha_f,$agegroup,$perpage,intval($offset)));
        }
        else {
            $sql = 'select c.*, p.* 
                from persona c, participacion p 
                where c.dni=p.dniCompetidor and (p.fecha_participacion BETWEEN ? and ?) and p.idPeso like ?  
                limit ? OFFSET ?';
            $query = $this->db->query($sql, array($fecha_i, $fecha_f, $weight, $perpage, intval($offset)));
        }

        return $query->result();
    }

    /**
     * Funcion que devuelve el numero de competidores en una determinada franja de tiempo ,peso y grupo de edad
     * @param $fecha_i fecha de inicio de busqueda
     * @param $fecha_f fecha de fin de busqueda
     * @param $weight Identificador del peso
     * @param $agegroup Identificador de grupo de edad
     * @return Int numero de competidores
     */
    public function countRowsCompetitorsBy($fecha_i, $fecha_f, $weight,$agegroup) {
        if($weight == self::ALL_WEIGHTS) {
            $sql = 'select c.*, p.*
                from persona c, participacion p ,peso pe 
                where c.dni=p.dniCompetidor and (p.fecha_participacion BETWEEN ? and ?) and p.idPeso = pe.idPeso and pe.idGrupoEdad = ? group by c.dni ';
            $query = $this->db->query($sql, array($fecha_i, $fecha_f,$agegroup));
        }
        else {
            $sql = 'select c.*, p.* 
                from persona c, participacion p 
                where c.dni=p.dniCompetidor and (p.fecha_participacion BETWEEN ? and ?) and p.idPeso = ? group by c.dni ';
            $query = $this->db->query($sql, array($fecha_i, $fecha_f, $weight));
        }


        return $query->num_rows();
    }

    /**
     * Participaciones de competidores entre dos fechas indicando peso y preparado para
     * la paginacion
     * @param $fecha_i fecha de inicio de busqueda
     * @param $fecha_f fecha de fin de busqueda
     * @param $weight Identificador del peso
     * @param $perpage Numero de competidores por pagina
     * @param $offset Desplazamiento para la base de datos
     * @param $agegroup Identificador de grupo de edad
     * @return mixed
     */
    public function getCompetitorsBy($fecha_i, $fecha_f, $weight, $perpage, $offset,$agegroup)
    {
        if($weight == self::ALL_WEIGHTS) {
            $sql = 'select c.*, p.*
                from persona c, participacion p ,peso pe 
                where c.dni=p.dniCompetidor and (p.fecha_participacion BETWEEN ? and ?) and p.idPeso = pe.idPeso and pe.idGrupoEdad = ? group by c.dni 
                limit ? OFFSET ?';
            $offset = $offset == null ? 0 : $offset;
            $query = $this->db->query($sql, array($fecha_i, $fecha_f,$agegroup,$perpage,intval($offset)));
        }
        else {
            $sql = 'select c.*, p.* 
                from persona c, participacion p 
                where c.dni=p.dniCompetidor and (p.fecha_participacion BETWEEN ? and ?) and p.idPeso like ? group by c.dni 
                limit ? OFFSET ?';
            $query = $this->db->query($sql, array($fecha_i, $fecha_f, $weight, $perpage, intval($offset)));
        }

        return $query->result();
    }

    /**
     * Devuelve tanto competidores como arbitros
     * @return Persona [] con todos sus campos
     */
    public function getPeople()
    {
        $sql = 'SELECT C.* 
                FROM Persona C
                WHERE C.dni';

        $query = $this->db->query($sql);
        return $query->result();
    }

    /**
     * Devuelve todos los competidores
     * @return Persona []
     */
    public function getAllCompetitors()
    {
        $sql = 'SELECT C.*, P.* 
                FROM Persona C, Participacion P 
                WHERE C.dni = P.dniCompetidor
                GROUP BY C.dni';
        $query = $this->db->query($sql);
        return $query->result();
    }


    /**
     * Devuelve todos los competidores delimitado por numero por pagina y desplazamiento
     * @param $perpage Numero de competidores por pagina
     * @param $segment_uri desplazamiento
     * @return Persona [] con todos sus campos
     */
    public function getCompetitors($perpage, $segment_uri)
    {
//           $query = $this -> db -> get('persona',$perpage,$segment_uri);
        $sql = 'SELECT C.*, P.* 
                FROM Persona C, Participacion P 
                WHERE C.dni = P.dniCompetidor
                LIMIT ? OFFSET ?';
        $segment_uri = $segment_uri == null ? 0 : $segment_uri;
        $q = $this->db->query($sql, array($perpage, intval($segment_uri)));
        return $q->result();
    }

    /**
     * Devuelve un competidor si existe
     * @param $name Nombre del competidor
     * @param $surname Primer apellido del competidor
     * @return Persona
     */
    public function getCompetitor($name, $surname)
    {
        $sql = 'SELECT P.* FROM Persona P WHERE P.nombre = ? AND P.apellido1 = ?';
        $query = $this->db->query($sql, array($name, $surname));

        return $query->row();
    }

    /**
     * Funcion que devuelve todos los dnis de las personas tanto arbitros como competidores
     * @return string []
     */
    public function getAllDnis()
    {
        $sql = 'SELECT P.dni FROM Persona P ORDER BY P.nombre ASC';
        $query = $this->db->query($sql);
        return $query->result();
    }

    /**
     * Devuelve un competidor si existe en funcion de su dni
     * @param $dni Identificador del competidor
     * @return Persona con todos sus campos
     */
    public function getCompetitorByDni($dni)
    {
        $sql = 'SELECT P.* FROM Persona P WHERE P.dni = ?';
        $query = $this->db->query($sql, array($dni));

        return $query->row();
    }

    /**
     * Funcion que crea un competidor
     * @param $competitor Comeptitor
     */
    public function createCompetitor($competitor)
    {
        $c = array(
            'dni' => $competitor->getDni(),
            'nombre' => $competitor->getNombre(),
            'apellido1' => $competitor->getApellido1(),
            'apellido2' => $competitor->getApellido2(),
            'paisISOA3' => $competitor->getPaisISOA3(),
            'autonomia' => $competitor->getAutonomia(),
            'provincia' => $competitor->getProvincia(),
            'email' => $competitor->getEmail()

        );
        $this->db->insert(self::TABLE_NAME, $c);
    }

    /**
     * Funcion que crea una inscricpion
     * @param $dni Identificador de la persona que se quiere inscribir
     * @param $idCompetition Identificador de la competicion
     * @param $teamName Nombre del equipo que se inscribe
     * @param $date Fecha en la que se realiza la inscripcion
     */
    public function createInscription($dni, $idCompetition, $teamName, $date)
    {
        $inscription = array(
            'dniPersona' => $dni,
            'nombreEquipo' => $teamName,
            'fecha_inscripcion' => $date,
            'idCompeticion' => $idCompetition,
            'arbitra' => 0
        );
        $this->db->insert('Inscripcion', $inscription);
    }

    /**
     * Funcion que crea una participacion
     * @param $dni Identificador de la persona
     * @param $idWeight Identificador del peso que se quiere realizar la inscripcion
     * @param $competition_date Fecha en la que se desarrollara la competicion
     */
    public function createParticipation($dni, $idWeight, $competition_date)
    {
        $participation = array(

            'idPeso' => $idWeight,
            'dniCompetidor' => $dni,
            'fecha_participacion' => $competition_date
        );
        $this->db->insert('Participacion', $participation);
    }

    /**
     * Funcion que devuelve una inscricpcion
     * @param $dni Identificador de la persona
     * @param $idCompetition Identificador de la competicion
     * @return Inscripcion
     */
    public function getInscription($dni, $idCompetition)
    {
        $sql = 'SELECT I.* FROM Inscripcion I WHERE I.dniPersona = ? AND I.idCompeticion = ?';
        $query = $this->db->query($sql, array($dni, $idCompetition));

        return $query->row();

    }

    /**
     * Funcion que devuelve los competidores en funcion de los parametros
     * @param $idWeight Identificador del peso
     * @param $competition_date Fecha de la competicion
     * @param $idCompetition Identificador de la competicion
     * @return Persona []
     */
    public function getCompetitorsByIdWeight($idWeight, $competition_date, $idCompetition)
    {
        $sql = 'SELECT DISTINCT C.*,I.nombreEquipo
                FROM Persona C, Participacion P,Inscripcion I, Peso Pe
                WHERE C.dni = I.dniPersona AND 
                I.idCompeticion = ? AND 
                I.arbitra = 0 AND 
                P.idPeso = ? AND 
                Pe.idPeso = p.idPeso AND
                P.dniCompetidor = C.dni AND
                P.fecha_participacion = ?';
        $query = $this->db->query($sql, array($idCompetition, $idWeight, $competition_date));

        return $query->result();

    }

    /**
     * Funcion para actualizar la participacion
     * @param $dni Identificador de la persona que se quiere actualizar la participacion
     * @param $date Fecha de la participacion
     * @param $idWeight Identificador del peso nuevo
     * @param $oldWeight Identificador del peso antiguo
     */
    public function updateParticipation($dni, $date, $idWeight, $oldWeight)
    {
        if ($this->existsParticipation($dni, $date, $oldWeight)) {
            $this->deleteParticipation($oldWeight, $date, $dni);
        }
        $data = array(
            'dniCompetidor' => $dni,
            'fecha_participacion' => $date,
            'idPeso' => $idWeight
        );

        $this->db->insert('Participacion', $data);

    }

    /**
     * Funcion que indica si existe o no una participacion en funcion de los
     * parametros dados
     * @param $dni Identificador de la persona
     * @param $date Fecha de participacion
     * @param $idWeight Identificador de peso
     * @return bool true en caso de que exista una participacion false en caso contrario
     */
    public function existsParticipation($dni, $date, $idWeight)
    {
        $sql = 'SELECT * 
                FROM Participacion P
                WHERE P.fecha_participacion = ? AND P.dniCompetidor = ? AND P.idPeso = ?';
        $query = $this->db->query($sql, array($date, $dni, $idWeight));

        return $query->num_rows() != 0;
    }

    /**
     * Funcion para eliminar la inscripcion de un arbito o competidor
     * @param $idCompetition Identficidador de la competicion
     * @param $dni Identificador de la persona
     */
    public function deleteInscription($idCompetition, $dni)
    {
        $this->db->where('idCompeticion', $idCompetition);
        $this->db->where('dniPersona', $dni);


        $this->db->delete('Inscripcion');
    }

    /**
     * Funcion que se utiliza para eliminar la participacion de un competidor en una competicion
     * en funcion de los parametros dados
     * @param $idWeight Identificador del peso
     * @param $date fecha de la competicion
     * @param $dni Identificador de la persona
     */
    public function deleteParticipation($idWeight, $date, $dni)
    {
        $this->db->where('idPeso', $idWeight);
        $this->db->where('dniCompetidor', $dni);
        $this->db->where('fecha_participacion', $date);

        $this->db->delete('Participacion');


    }




}